package hr.algebra.permissions

import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.telephony.SmsManager
import android.text.Editable
import android.text.TextWatcher
import android.view.View
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import hr.algebra.permissions.databinding.ActivityMainBinding
import androidx.core.net.toUri


private const val PHONE_REQ = 1
private const val SMS_REQ = 2

class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityMainBinding

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setupListeners()
    }

    private fun setupListeners() {
        val textWatcher = object: TextWatcher {
            override fun afterTextChanged(p0: Editable?) {

            }

            override fun beforeTextChanged(
                p0: CharSequence?,
                p1: Int,
                p2: Int,
                p3: Int,
            ) {
            }

            override fun onTextChanged(
                p0: CharSequence?,
                p1: Int,
                p2: Int,
                p3: Int,
            ) {
                val canCall = binding.etNumber.text.isNotBlank()
                val canSend = canCall && binding.etMessage.text.isNotBlank()

                binding.btnPhone.visibility = if(canCall) View.VISIBLE else View.INVISIBLE
                binding.btnPhoneDirect.visibility = if(canCall) View.VISIBLE else View.INVISIBLE
                binding.btnSms.visibility = if(canSend) View.VISIBLE else View.INVISIBLE
                binding.btnSmsDirect.visibility = if(canSend) View.VISIBLE else View.INVISIBLE
            }
        }
        binding.etNumber.addTextChangedListener(textWatcher)
        binding.etMessage.addTextChangedListener(textWatcher)

        binding.btnPhone.setOnClickListener { call() }
        binding.btnPhoneDirect.setOnClickListener { callDirect() }

        binding.btnSms.setOnClickListener { sms() }
        binding.btnSmsDirect.setOnClickListener { smsDirect() }
    }

    private fun call() {
        startActivity(
            Intent(
                Intent.ACTION_DIAL,
                "tel:${binding.etNumber.text}".toUri()
            )
        )
    }

    private fun callDirect() {
        if (ActivityCompat.checkSelfPermission(this, android.Manifest.permission.CALL_PHONE)
            != PackageManager.PERMISSION_GRANTED
        ) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf(android.Manifest.permission.CALL_PHONE),
                PHONE_REQ
            )
            return
        }
        startActivity(
            Intent(
                Intent.ACTION_CALL,
                "tel:${binding.etNumber.text}".toUri()
            )
        )
    }

    private fun sms() {
        startActivity(
            Intent(
                Intent.ACTION_SENDTO,
                "smsto:${binding.etNumber.text}".toUri()
            ).apply {
                putExtra("sms_body", binding.etMessage.text.toString())
            }
        )
    }

    private fun smsDirect() {
        if (ActivityCompat.checkSelfPermission(this, android.Manifest.permission.SEND_SMS)
            != PackageManager.PERMISSION_GRANTED
        ) {
            ActivityCompat.requestPermissions(
                this,
                arrayOf(android.Manifest.permission.SEND_SMS),
                SMS_REQ
            )
            return
        }

        val smsManager = if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            getSystemService(SmsManager::class.java)
        } else {
            @Suppress("DEPRECATION")
            SmsManager.getDefault()
        }

        smsManager.sendTextMessage(
            binding.etNumber.text.toString(),
            null,
            binding.etMessage.text.toString().trim(),
            null,
            null
        )

    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
            when(requestCode) {
                PHONE_REQ -> callDirect()
                SMS_REQ -> smsDirect()
            }
        }
    }

}