package hr.algebra.networking.tcp;

import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.nio.file.Files;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;

public class HttpServer {
    // copy from http_server.txt
    private static final int PORT = 80;
    private static final String WELCOME_FILE = "index.html";
    private static final String EXT_HTML = ".html";
    private static final String EXT_JPG = ".jpg";

    public static void main(String[] args) {
        acceptRequests();
    }

    private static void acceptRequests() {
        try (ServerSocket serverSocket = new ServerSocket(PORT)) {
            System.err.println("Server listening on port:" + serverSocket.getLocalPort());
            while (true) {
                Socket clientSocket = serverSocket.accept(); // client get assigned a port so that the system knows where to return
                System.err.println("Client connected from port:" +clientSocket.getPort());

                new Thread(() -> processHttp(clientSocket)).start();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void processHttp(Socket clientSocket) {
        try (BufferedReader br = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()))) {
            String filename = WELCOME_FILE;

            String line;
            while ((line = br.readLine()) != null && !line.isEmpty()) {
                if (line.startsWith("GET")) {

                    filename = line.substring(line.indexOf("/") + 1, line.lastIndexOf(" "));
                    // enable friendly urls
                    // if filename has extension, leave it, otherwise apend ".html"
                    filename = filename.contains(".") ? filename : filename + EXT_HTML; //filename.concat(EXT_HTML)
                    System.out.println("filename: " + filename);
                    break; // at the end of debug

                }
                // inspect headers first -> localhost/test.html GET /test.html HTTP/1.1
                //System.out.println(line);
            }

            sendResponse(clientSocket, new File(filename));

        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void sendResponse(Socket clientSocket, File file) throws IOException {
        // ensure default
        if (!file.exists()) {
            file = new File(WELCOME_FILE);
        }
        if (file.toString().endsWith(EXT_JPG)) {
            flushJpg(clientSocket, file);
        } else {
            flushText(clientSocket, file);
        }
    }

    private static void flushJpg(Socket clientSocket, File file) throws IOException {
        try (OutputStream os = new BufferedOutputStream(clientSocket.getOutputStream())) {
            os.write(Files.readAllBytes(file.toPath())); // path -> file and vice versa - io to nio good practice
            //os.flush();
        }
    }

    private static void flushText(Socket clientSocket, File file) throws IOException {
        try (Writer w = new BufferedWriter(new OutputStreamWriter(clientSocket.getOutputStream()))) {
            if (file.toString().endsWith(EXT_HTML)) {
                addHeader(file, w);
            }
            addBody(file, w);
            //bw.flush();
        }
    }

    // copy from https://www.tutorialspoint.com/http/http_responses (copy method from resources)
    // alt + shift + arrow keys
    private static void addHeader(File file, Writer w) throws IOException {
        w.write("HTTP/1.1 200 OK\n\r");
        w.write("Date: " + ZonedDateTime.now().format(DateTimeFormatter.RFC_1123_DATE_TIME) + "\n\r");
        w.write("Server: MiliczasServant/2.2.14 (Win64)\n\r");
        w.write(" Last-Modified: " + ZonedDateTime.now().minusDays(2).format(DateTimeFormatter.RFC_1123_DATE_TIME) + "\n\r");
        // very important
        w.write("Content-Length: "  + file.length() + "\n\r");
        w.write("Content-Type: "+ Files.probeContentType(file.toPath())+"; charset=utf-8\n\r");
        w.write("Connection: Closed\n\r");
        // very important
        w.write("\n\r");
    }

    private static void addBody(File file, Writer w) throws IOException {
        // Files.readAllLines() closes the stream so we are ok without try finally!
        // we cannot use lambda because it throws IOException
        for (String line : Files.readAllLines(file.toPath())) {
            w.write(line + "\n\r");
        }
    }

}
